/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import org.eclipse.rdf4j.common.io.ByteArrayUtil;
import org.eclipse.rdf4j.sail.SailException;
import org.eclipse.rdf4j.sail.nativerdf.InMemRecordCache;
import org.eclipse.rdf4j.sail.nativerdf.RecordCache;
import org.eclipse.rdf4j.sail.nativerdf.SequentialRecordCache;
import org.eclipse.rdf4j.sail.nativerdf.SortedRecordCache;
import org.eclipse.rdf4j.sail.nativerdf.TxnStatusFile;
import org.eclipse.rdf4j.sail.nativerdf.btree.BTree;
import org.eclipse.rdf4j.sail.nativerdf.btree.RecordComparator;
import org.eclipse.rdf4j.sail.nativerdf.btree.RecordIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TripleStore
implements Closeable {
    private static final int CHECK_MEMORY_PRESSURE_INTERVAL = TripleStore.isAssertionsEnabled() ? 3 : 1024;
    private static final long MIN_FREE_MEMORY_BEFORE_OVERFLOW = TripleStore.isAssertionsEnabled() ? Long.MAX_VALUE : 0x8000000L;
    private static final String DEFAULT_INDEXES = "spoc,posc";
    private static final String PROPERTIES_FILE = "triples.prop";
    private static final String VERSION_KEY = "version";
    private static final String INDEXES_KEY = "triple-indexes";
    private static final int SCHEME_VERSION = 10;
    static final int RECORD_LENGTH = 17;
    static final int SUBJ_IDX = 0;
    static final int PRED_IDX = 4;
    static final int OBJ_IDX = 8;
    static final int CONTEXT_IDX = 12;
    static final int FLAG_IDX = 16;
    static final byte EXPLICIT_FLAG = 1;
    static final byte ADDED_FLAG = 2;
    static final byte REMOVED_FLAG = 4;
    static final byte TOGGLE_EXPLICIT_FLAG = 8;
    private static final Logger logger = LoggerFactory.getLogger(TripleStore.class);
    private final File dir;
    private final Properties properties;
    private final List<TripleIndex> indexes = new ArrayList<TripleIndex>();
    private final boolean forceSync;
    private final TxnStatusFile txnStatusFile;
    private volatile SortedRecordCache updatedTriplesCache;

    public TripleStore(File dir, String indexSpecStr) throws IOException, SailException {
        this(dir, indexSpecStr, false);
    }

    public TripleStore(File dir, String indexSpecStr, boolean forceSync) throws IOException, SailException {
        this.dir = dir;
        this.forceSync = forceSync;
        this.txnStatusFile = new TxnStatusFile(dir);
        File propFile = new File(dir, PROPERTIES_FILE);
        if (!propFile.exists()) {
            this.properties = new Properties();
            Set<String> indexSpecs = this.parseIndexSpecList(indexSpecStr);
            if (indexSpecs.isEmpty()) {
                logger.debug("No indexes specified, using default indexes: {}", (Object)DEFAULT_INDEXES);
                indexSpecStr = DEFAULT_INDEXES;
                indexSpecs = this.parseIndexSpecList(indexSpecStr);
            }
            this.initIndexes(indexSpecs);
        } else {
            this.properties = this.loadProperties(propFile);
            this.checkVersion();
            Set<String> indexSpecs = this.getIndexSpecs();
            this.initIndexes(indexSpecs);
            TxnStatusFile.TxnStatus txnStatus = this.txnStatusFile.getTxnStatus();
            if (txnStatus == TxnStatusFile.TxnStatus.NONE) {
                logger.trace("No uncompleted transactions found");
            } else {
                this.processUncompletedTransaction(txnStatus);
            }
            Set<String> reqIndexSpecs = this.parseIndexSpecList(indexSpecStr);
            if (reqIndexSpecs.isEmpty()) {
                indexSpecStr = this.properties.getProperty(INDEXES_KEY);
            } else if (!reqIndexSpecs.equals(indexSpecs)) {
                this.reindex(indexSpecs, reqIndexSpecs);
            }
        }
        if (!String.valueOf(10).equals(this.properties.getProperty(VERSION_KEY)) || !indexSpecStr.equals(this.properties.getProperty(INDEXES_KEY))) {
            this.properties.setProperty(VERSION_KEY, String.valueOf(10));
            this.properties.setProperty(INDEXES_KEY, indexSpecStr);
            this.storeProperties(propFile);
        }
    }

    private void checkVersion() throws SailException {
        String versionStr = this.properties.getProperty(VERSION_KEY);
        if (versionStr == null) {
            logger.warn("{} missing in TripleStore's properties file", (Object)VERSION_KEY);
        } else {
            try {
                int version = Integer.parseInt(versionStr);
                if (version < 10) {
                    throw new SailException("Directory contains incompatible triple data");
                }
                if (version > 10) {
                    throw new SailException("Directory contains data that uses a newer data format");
                }
            }
            catch (NumberFormatException e) {
                logger.warn("Malformed version number in TripleStore's properties file");
            }
        }
    }

    private Set<String> getIndexSpecs() throws SailException {
        String indexesStr = this.properties.getProperty(INDEXES_KEY);
        if (indexesStr == null) {
            throw new SailException("triple-indexes missing in TripleStore's properties file");
        }
        Set<String> indexSpecs = this.parseIndexSpecList(indexesStr);
        if (indexSpecs.isEmpty()) {
            throw new SailException("No triple-indexes found in TripleStore's properties file");
        }
        return indexSpecs;
    }

    private Set<String> parseIndexSpecList(String indexSpecStr) throws SailException {
        HashSet<String> indexes = new HashSet<String>();
        if (indexSpecStr != null) {
            StringTokenizer tok = new StringTokenizer(indexSpecStr, ", \t");
            while (tok.hasMoreTokens()) {
                String index = tok.nextToken().toLowerCase();
                if (index.length() != 4 || index.indexOf(115) == -1 || index.indexOf(112) == -1 || index.indexOf(111) == -1 || index.indexOf(99) == -1) {
                    throw new SailException("invalid value '" + index + "' in index specification: " + indexSpecStr);
                }
                indexes.add(index);
            }
        }
        return indexes;
    }

    private void initIndexes(Set<String> indexSpecs) throws IOException {
        for (String fieldSeq : indexSpecs) {
            logger.trace("Initializing index '{}'...", (Object)fieldSeq);
            this.indexes.add(new TripleIndex(fieldSeq));
        }
    }

    private void processUncompletedTransaction(TxnStatusFile.TxnStatus txnStatus) throws IOException {
        switch (txnStatus) {
            case COMMITTING: {
                logger.info("Detected uncompleted commit, trying to complete");
                try {
                    this.commit();
                    logger.info("Uncompleted commit completed successfully");
                    break;
                }
                catch (IOException e) {
                    logger.error("Failed to restore from uncompleted commit", (Throwable)e);
                    throw e;
                }
            }
            case ROLLING_BACK: {
                logger.info("Detected uncompleted rollback, trying to complete");
                try {
                    this.rollback();
                    logger.info("Uncompleted rollback completed successfully");
                    break;
                }
                catch (IOException e) {
                    logger.error("Failed to restore from uncompleted rollback", (Throwable)e);
                    throw e;
                }
            }
            case ACTIVE: {
                logger.info("Detected unfinished transaction, trying to roll back");
                try {
                    this.rollback();
                    logger.info("Unfinished transaction rolled back successfully");
                    break;
                }
                catch (IOException e) {
                    logger.error("Failed to roll back unfinished transaction", (Throwable)e);
                    throw e;
                }
            }
            case UNKNOWN: {
                logger.info("Read invalid or unknown transaction status, trying to roll back");
                try {
                    this.rollback();
                    logger.info("Successfully performed a rollback for invalid or unknown transaction status");
                    break;
                }
                catch (IOException e) {
                    logger.error("Failed to perform rollback for invalid or unknown transaction status", (Throwable)e);
                    throw e;
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void reindex(Set<String> currentIndexSpecs, Set<String> newIndexSpecs) throws IOException, SailException {
        HashMap<String, TripleIndex> currentIndexes = new HashMap<String, TripleIndex>();
        for (TripleIndex index : this.indexes) {
            currentIndexes.put(new String(index.getFieldSeq()), index);
        }
        HashSet<String> addedIndexSpecs = new HashSet<String>(newIndexSpecs);
        addedIndexSpecs.removeAll(currentIndexSpecs);
        if (!addedIndexSpecs.isEmpty()) {
            TripleIndex sourceIndex = this.indexes.get(0);
            for (String fieldSeq : addedIndexSpecs) {
                logger.debug("Initializing new index '{}'...", (Object)fieldSeq);
                TripleIndex addedIndex = new TripleIndex(fieldSeq);
                BTree addedBTree = null;
                RecordIterator sourceIter = null;
                try {
                    byte[] value;
                    addedBTree = addedIndex.getBTree();
                    sourceIter = sourceIndex.getBTree().iterateAll();
                    while ((value = sourceIter.next()) != null) {
                        addedBTree.insert(value);
                    }
                }
                finally {
                    try {
                        if (sourceIter != null) {
                            sourceIter.close();
                        }
                    }
                    finally {
                        if (addedBTree != null) {
                            addedBTree.sync();
                        }
                    }
                }
                currentIndexes.put(fieldSeq, addedIndex);
            }
            logger.debug("New index(es) initialized");
        }
        HashSet<String> removedIndexSpecs = new HashSet<String>(currentIndexSpecs);
        removedIndexSpecs.removeAll(newIndexSpecs);
        ArrayList<Throwable> removedIndexExceptions = new ArrayList<Throwable>();
        for (String fieldSeq : removedIndexSpecs) {
            try {
                TripleIndex removedIndex = (TripleIndex)currentIndexes.remove(fieldSeq);
                boolean deleted = removedIndex.getBTree().delete();
                if (deleted) {
                    logger.debug("Deleted file(s) for removed {} index", (Object)fieldSeq);
                    continue;
                }
                logger.warn("Unable to delete file(s) for removed {} index", (Object)fieldSeq);
            }
            catch (Throwable e) {
                removedIndexExceptions.add(e);
            }
        }
        if (!removedIndexExceptions.isEmpty()) {
            throw new IOException((Throwable)removedIndexExceptions.get(0));
        }
        this.indexes.clear();
        for (String fieldSeq : newIndexSpecs) {
            this.indexes.add((TripleIndex)currentIndexes.remove(fieldSeq));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        try {
            ArrayList<Throwable> caughtExceptions = new ArrayList<Throwable>();
            for (TripleIndex index : this.indexes) {
                try {
                    index.getBTree().close();
                }
                catch (Throwable e) {
                    logger.warn("Failed to close file for {} index", (Object)new String(index.getFieldSeq()));
                    caughtExceptions.add(e);
                }
            }
            if (!caughtExceptions.isEmpty()) {
                throw new IOException((Throwable)caughtExceptions.get(0));
            }
        }
        finally {
            try {
                this.txnStatusFile.close();
            }
            finally {
                SortedRecordCache toCloseUpdatedTriplesCache = this.updatedTriplesCache;
                this.updatedTriplesCache = null;
                if (toCloseUpdatedTriplesCache != null) {
                    toCloseUpdatedTriplesCache.discard();
                }
            }
        }
    }

    public RecordIterator getTriples(int subj, int pred, int obj, int context) {
        return this.getTriples(subj, pred, obj, context, 0, 2);
    }

    public RecordIterator getTriples(int subj, int pred, int obj, int context, boolean readTransaction) {
        if (readTransaction) {
            return this.getTriples(subj, pred, obj, context, 0, 4);
        }
        return this.getTriples(subj, pred, obj, context, 0, 2);
    }

    public RecordIterator getAllTriplesSortedByContext(boolean readTransaction) {
        if (readTransaction) {
            return this.getAllTriplesSortedByContext(0, 4);
        }
        return this.getAllTriplesSortedByContext(0, 2);
    }

    public RecordIterator getTriples(int subj, int pred, int obj, int context, boolean explicit, boolean readTransaction) {
        int flags = 0;
        int flagsMask = 0;
        if (readTransaction) {
            flagsMask |= 4;
        } else {
            flagsMask |= 2;
            if (explicit) {
                flags |= 1;
                flagsMask |= 1;
            }
        }
        RecordIterator btreeIter = this.getTriples(subj, pred, obj, context, flags, flagsMask);
        if (readTransaction && explicit) {
            btreeIter = new ExplicitStatementFilter(btreeIter);
        } else if (!explicit) {
            btreeIter = new ImplicitStatementFilter(btreeIter);
        }
        return btreeIter;
    }

    private RecordIterator getTriples(int subj, int pred, int obj, int context, int flags, int flagsMask) {
        TripleIndex index = this.getBestIndex(subj, pred, obj, context);
        boolean doRangeSearch = index.getPatternScore(subj, pred, obj, context) > 0;
        return this.getTriplesUsingIndex(subj, pred, obj, context, flags, flagsMask, index, doRangeSearch);
    }

    private RecordIterator getAllTriplesSortedByContext(int flags, int flagsMask) {
        for (TripleIndex index : this.indexes) {
            if (index.getFieldSeq()[0] != 'c') continue;
            return this.getTriplesUsingIndex(-1, -1, -1, -1, flags, flagsMask, index, false);
        }
        return null;
    }

    private RecordIterator getTriplesUsingIndex(int subj, int pred, int obj, int context, int flags, int flagsMask, TripleIndex index, boolean rangeSearch) {
        byte[] searchKey = this.getSearchKey(subj, pred, obj, context, flags);
        byte[] searchMask = this.getSearchMask(subj, pred, obj, context, flagsMask);
        if (rangeSearch) {
            byte[] minValue = this.getMinValue(subj, pred, obj, context);
            byte[] maxValue = this.getMaxValue(subj, pred, obj, context);
            return index.getBTree().iterateRangedValues(searchKey, searchMask, minValue, maxValue);
        }
        return index.getBTree().iterateValues(searchKey, searchMask);
    }

    protected double cardinality(int subj, int pred, int obj, int context) throws IOException {
        double rangeSize;
        TripleIndex index = this.getBestIndex(subj, pred, obj, context);
        BTree btree = index.btree;
        if (index.getPatternScore(subj, pred, obj, context) == 0) {
            rangeSize = btree.getValueCountEstimate();
        } else {
            byte[] minValue = this.getMinValue(subj, pred, obj, context);
            byte[] maxValue = this.getMaxValue(subj, pred, obj, context);
            rangeSize = btree.getValueCountEstimate(minValue, maxValue);
        }
        return rangeSize;
    }

    protected TripleIndex getBestIndex(int subj, int pred, int obj, int context) {
        int bestScore = -1;
        TripleIndex bestIndex = null;
        for (TripleIndex index : this.indexes) {
            int score = index.getPatternScore(subj, pred, obj, context);
            if (score <= bestScore) continue;
            bestScore = score;
            bestIndex = index;
        }
        return bestIndex;
    }

    public void clear() throws IOException {
        for (TripleIndex index : this.indexes) {
            index.getBTree().clear();
        }
    }

    public boolean storeTriple(int subj, int pred, int obj, int context) throws IOException {
        return this.storeTriple(subj, pred, obj, context, true);
    }

    public boolean storeTriple(int subj, int pred, int obj, int context, boolean explicit) throws IOException {
        boolean stAdded;
        byte[] data = this.getData(subj, pred, obj, context, 0);
        byte[] storedData = this.indexes.get(0).getBTree().get(data);
        if (storedData == null) {
            data[16] = (byte)(data[16] | 2);
            if (explicit) {
                data[16] = (byte)(data[16] | 1);
            }
            stAdded = true;
        } else {
            boolean wasToggled;
            byte flags = storedData[16];
            boolean wasExplicit = (flags & 1) != 0;
            boolean wasAdded = (flags & 2) != 0;
            boolean wasRemoved = (flags & 4) != 0;
            boolean bl = wasToggled = (flags & 8) != 0;
            if (wasAdded) {
                data[16] = (byte)(data[16] | 2);
                if (explicit || wasExplicit) {
                    data[16] = (byte)(data[16] | 1);
                }
            } else {
                if (wasExplicit) {
                    data[16] = (byte)(data[16] | 1);
                }
                if (explicit) {
                    if (!wasExplicit) {
                        data[16] = (byte)(data[16] | 8);
                    }
                } else if (wasRemoved) {
                    if (wasExplicit) {
                        data[16] = (byte)(data[16] | 8);
                    }
                } else if (wasToggled) {
                    data[16] = (byte)(data[16] | 8);
                }
            }
            stAdded = wasRemoved;
        }
        if (storedData == null || !Arrays.equals(data, storedData)) {
            for (TripleIndex index : this.indexes) {
                index.getBTree().insert(data);
            }
            this.updatedTriplesCache.storeRecord(data);
        }
        return stAdded;
    }

    @Deprecated(since="2.5.3")
    public int removeTriples(int subj, int pred, int obj, int context) throws IOException {
        Map<Integer, Long> countPerContext = this.removeTriplesByContext(subj, pred, obj, context);
        return (int)countPerContext.values().stream().mapToLong(Long::longValue).sum();
    }

    public Map<Integer, Long> removeTriplesByContext(int subj, int pred, int obj, int context) throws IOException {
        RecordIterator iter = this.getTriples(subj, pred, obj, context, 0, 0);
        return this.removeTriples(iter);
    }

    @Deprecated(since="2.5.3")
    public int removeTriples(int subj, int pred, int obj, int context, boolean explicit) throws IOException {
        Map<Integer, Long> countPerContext = this.removeTriplesByContext(subj, pred, obj, context, explicit);
        return (int)countPerContext.values().stream().mapToLong(Long::longValue).sum();
    }

    public Map<Integer, Long> removeTriplesByContext(int subj, int pred, int obj, int context, boolean explicit) throws IOException {
        int flags = explicit ? 1 : 0;
        try (RecordIterator iter = this.getTriples(subj, pred, obj, context, flags, 1);){
            Map<Integer, Long> map = this.removeTriples(iter);
            return map;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<Integer, Long> removeTriples(RecordIterator iter) throws IOException {
        byte[] data = iter.next();
        if (data == null) {
            return Collections.emptyMap();
        }
        HashMap<Integer, Long> perContextCounts = new HashMap<Integer, Long>();
        RecordCache removedTriplesCache = new InMemRecordCache();
        try {
            try (RecordIterator recordIterator = iter;){
                while (data != null) {
                    if ((data[16] & 4) == 0) {
                        data[16] = (byte)(data[16] | 4);
                        removedTriplesCache.storeRecord(data);
                        int context = ByteArrayUtil.getInt((byte[])data, (int)12);
                        perContextCounts.merge(context, 1L, Long::sum);
                    }
                    data = iter.next();
                    if (!this.shouldOverflowToDisk(removedTriplesCache)) continue;
                    logger.debug("Overflowing RecordCache to disk due to low free mem.");
                    assert (removedTriplesCache instanceof InMemRecordCache);
                    InMemRecordCache old = removedTriplesCache;
                    removedTriplesCache = new SequentialRecordCache(this.dir, 17);
                    removedTriplesCache.storeRecords(old);
                    old.clear();
                }
            }
            this.updatedTriplesCache.storeRecords(removedTriplesCache);
            for (TripleIndex index : this.indexes) {
                BTree btree = index.getBTree();
                RecordIterator recIter = removedTriplesCache.getRecords();
                try {
                    while ((data = recIter.next()) != null) {
                        btree.insert(data);
                    }
                }
                finally {
                    if (recIter == null) continue;
                    recIter.close();
                }
            }
        }
        finally {
            removedTriplesCache.discard();
        }
        return perContextCounts;
    }

    private boolean shouldOverflowToDisk(RecordCache removedTriplesCache) {
        if (removedTriplesCache instanceof InMemRecordCache && removedTriplesCache.getRecordCount() % (long)CHECK_MEMORY_PRESSURE_INTERVAL == 0L) {
            Runtime runtime = Runtime.getRuntime();
            long allocatedMemory = runtime.totalMemory() - runtime.freeMemory();
            long presumableFreeMemory = runtime.maxMemory() - allocatedMemory;
            logger.trace("Free memory {} MB and required free memory {} MB", (Object)(presumableFreeMemory / 1024L / 1024L), (Object)(MIN_FREE_MEMORY_BEFORE_OVERFLOW / 1024L / 1024L));
            return presumableFreeMemory < MIN_FREE_MEMORY_BEFORE_OVERFLOW;
        }
        return false;
    }

    public void startTransaction() throws IOException {
        this.txnStatusFile.setTxnStatus(TxnStatusFile.TxnStatus.ACTIVE);
        long maxRecords = this.indexes.get(0).getBTree().getValueCountEstimate() / 10L;
        if (this.updatedTriplesCache == null) {
            this.updatedTriplesCache = new SortedRecordCache(this.dir, 17, maxRecords, new TripleComparator("spoc"));
        } else {
            assert (this.updatedTriplesCache.getRecordCount() == 0L) : "updatedTripleCache should have been cleared upon commit or rollback";
            this.updatedTriplesCache.setMaxRecords(maxRecords);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void commit() throws IOException {
        this.txnStatusFile.setTxnStatus(TxnStatusFile.TxnStatus.COMMITTING);
        boolean validCache = this.updatedTriplesCache != null && this.updatedTriplesCache.isValid();
        for (TripleIndex index : this.indexes) {
            BTree btree = index.getBTree();
            try (RecordIterator iter = validCache ? this.updatedTriplesCache.getRecords() : btree.iterateAll();){
                byte[] data;
                while ((data = iter.next()) != null) {
                    boolean wasToggled;
                    byte flags = data[16];
                    boolean wasAdded = (flags & 2) != 0;
                    boolean wasRemoved = (flags & 4) != 0;
                    boolean bl = wasToggled = (flags & 8) != 0;
                    if (wasRemoved) {
                        btree.remove(data);
                        continue;
                    }
                    if (!wasAdded && !wasToggled) continue;
                    if (wasToggled) {
                        data[16] = (byte)(data[16] ^ 1);
                    }
                    if (wasAdded) {
                        data[16] = (byte)(data[16] ^ 2);
                    }
                    if (validCache) {
                        btree.insert(data);
                        continue;
                    }
                    iter.set(data);
                }
            }
        }
        if (this.updatedTriplesCache != null) {
            this.updatedTriplesCache.clear();
        }
        this.sync();
        this.txnStatusFile.setTxnStatus(TxnStatusFile.TxnStatus.NONE);
    }

    private void checkAllCommitted() throws IOException {
        for (TripleIndex index : this.indexes) {
            System.out.println("Checking " + index + " index");
            BTree btree = index.getBTree();
            RecordIterator iter = btree.iterateAll();
            try {
                byte[] data = iter.next();
                while (data != null) {
                    boolean wasToggled;
                    byte flags = data[16];
                    boolean wasAdded = (flags & 2) != 0;
                    boolean wasRemoved = (flags & 4) != 0;
                    boolean bl = wasToggled = (flags & 8) != 0;
                    if (wasAdded || wasRemoved || wasToggled) {
                        System.out.println("unexpected triple: " + ByteArrayUtil.toHexString((byte[])data));
                    }
                    data = iter.next();
                }
            }
            finally {
                if (iter == null) continue;
                iter.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void rollback() throws IOException {
        this.txnStatusFile.setTxnStatus(TxnStatusFile.TxnStatus.ROLLING_BACK);
        boolean validCache = this.updatedTriplesCache != null && this.updatedTriplesCache.isValid();
        int txnFlagsMask = -15;
        for (TripleIndex index : this.indexes) {
            BTree btree = index.getBTree();
            try (RecordIterator iter = validCache ? this.updatedTriplesCache.getRecords() : btree.iterateAll();){
                byte[] data;
                while ((data = iter.next()) != null) {
                    boolean wasToggled;
                    byte flags = data[16];
                    boolean wasAdded = (flags & 2) != 0;
                    boolean wasRemoved = (flags & 4) != 0;
                    boolean bl = wasToggled = (flags & 8) != 0;
                    if (wasAdded) {
                        btree.remove(data);
                        continue;
                    }
                    if (!wasRemoved && !wasToggled) continue;
                    data[16] = (byte)(data[16] & txnFlagsMask);
                    if (validCache) {
                        btree.insert(data);
                        continue;
                    }
                    iter.set(data);
                }
            }
        }
        if (this.updatedTriplesCache != null) {
            this.updatedTriplesCache.clear();
        }
        this.sync();
        this.txnStatusFile.setTxnStatus(TxnStatusFile.TxnStatus.NONE);
    }

    protected void sync() throws IOException {
        ArrayList<Throwable> exceptions = new ArrayList<Throwable>();
        for (TripleIndex index : this.indexes) {
            try {
                index.getBTree().sync();
            }
            catch (Throwable e) {
                exceptions.add(e);
            }
        }
        if (!exceptions.isEmpty()) {
            throw new IOException((Throwable)exceptions.get(0));
        }
    }

    private byte[] getData(int subj, int pred, int obj, int context, int flags) {
        byte[] data = new byte[17];
        ByteArrayUtil.putInt((int)subj, (byte[])data, (int)0);
        ByteArrayUtil.putInt((int)pred, (byte[])data, (int)4);
        ByteArrayUtil.putInt((int)obj, (byte[])data, (int)8);
        ByteArrayUtil.putInt((int)context, (byte[])data, (int)12);
        data[16] = (byte)flags;
        return data;
    }

    private byte[] getSearchKey(int subj, int pred, int obj, int context, int flags) {
        return this.getData(subj, pred, obj, context, flags);
    }

    private byte[] getSearchMask(int subj, int pred, int obj, int context, int flags) {
        byte[] mask = new byte[17];
        if (subj != -1) {
            ByteArrayUtil.putInt((int)-1, (byte[])mask, (int)0);
        }
        if (pred != -1) {
            ByteArrayUtil.putInt((int)-1, (byte[])mask, (int)4);
        }
        if (obj != -1) {
            ByteArrayUtil.putInt((int)-1, (byte[])mask, (int)8);
        }
        if (context != -1) {
            ByteArrayUtil.putInt((int)-1, (byte[])mask, (int)12);
        }
        mask[16] = (byte)flags;
        return mask;
    }

    private byte[] getMinValue(int subj, int pred, int obj, int context) {
        byte[] minValue = new byte[17];
        ByteArrayUtil.putInt((int)(subj == -1 ? 0 : subj), (byte[])minValue, (int)0);
        ByteArrayUtil.putInt((int)(pred == -1 ? 0 : pred), (byte[])minValue, (int)4);
        ByteArrayUtil.putInt((int)(obj == -1 ? 0 : obj), (byte[])minValue, (int)8);
        ByteArrayUtil.putInt((int)(context == -1 ? 0 : context), (byte[])minValue, (int)12);
        minValue[16] = 0;
        return minValue;
    }

    private byte[] getMaxValue(int subj, int pred, int obj, int context) {
        byte[] maxValue = new byte[17];
        ByteArrayUtil.putInt((int)(subj == -1 ? -1 : subj), (byte[])maxValue, (int)0);
        ByteArrayUtil.putInt((int)(pred == -1 ? -1 : pred), (byte[])maxValue, (int)4);
        ByteArrayUtil.putInt((int)(obj == -1 ? -1 : obj), (byte[])maxValue, (int)8);
        ByteArrayUtil.putInt((int)(context == -1 ? -1 : context), (byte[])maxValue, (int)12);
        maxValue[16] = -1;
        return maxValue;
    }

    private Properties loadProperties(File propFile) throws IOException {
        try (FileInputStream in = new FileInputStream(propFile);){
            Properties properties = new Properties();
            properties.load(in);
            Properties properties2 = properties;
            return properties2;
        }
    }

    private void storeProperties(File propFile) throws IOException {
        try (FileOutputStream out = new FileOutputStream(propFile);){
            this.properties.store(out, "triple indexes meta-data, DO NOT EDIT!");
        }
    }

    private static boolean isAssertionsEnabled() {
        try {
            assert (false);
            return false;
        }
        catch (AssertionError ignored) {
            return true;
        }
    }

    private static class TripleComparator
    implements RecordComparator {
        private final char[] fieldSeq;

        public TripleComparator(String fieldSeq) {
            this.fieldSeq = fieldSeq.toCharArray();
        }

        public char[] getFieldSeq() {
            return this.fieldSeq;
        }

        @Override
        public final int compareBTreeValues(byte[] key, byte[] data, int offset, int length) {
            for (char field : this.fieldSeq) {
                int fieldIdx;
                switch (field) {
                    case 's': {
                        fieldIdx = 0;
                        break;
                    }
                    case 'p': {
                        fieldIdx = 4;
                        break;
                    }
                    case 'o': {
                        fieldIdx = 8;
                        break;
                    }
                    case 'c': {
                        fieldIdx = 12;
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("invalid character '" + field + "' in field sequence: " + new String(this.fieldSeq));
                    }
                }
                int diff = ByteArrayUtil.compareRegion((byte[])key, (int)fieldIdx, (byte[])data, (int)(offset + fieldIdx), (int)4);
                if (diff == 0) continue;
                return diff;
            }
            return 0;
        }
    }

    private class TripleIndex {
        private final TripleComparator tripleComparator;
        private final BTree btree;

        public TripleIndex(String fieldSeq) throws IOException {
            this.tripleComparator = new TripleComparator(fieldSeq);
            this.btree = new BTree(TripleStore.this.dir, this.getFilenamePrefix(fieldSeq), 2048, 17, this.tripleComparator, TripleStore.this.forceSync);
        }

        private String getFilenamePrefix(String fieldSeq) {
            return "triples-" + fieldSeq;
        }

        public char[] getFieldSeq() {
            return this.tripleComparator.getFieldSeq();
        }

        public BTree getBTree() {
            return this.btree;
        }

        public int getPatternScore(int subj, int pred, int obj, int context) {
            int score = 0;
            block6: for (char field : this.tripleComparator.getFieldSeq()) {
                switch (field) {
                    case 's': {
                        if (subj >= 0) {
                            ++score;
                            continue block6;
                        }
                        return score;
                    }
                    case 'p': {
                        if (pred >= 0) {
                            ++score;
                            continue block6;
                        }
                        return score;
                    }
                    case 'o': {
                        if (obj >= 0) {
                            ++score;
                            continue block6;
                        }
                        return score;
                    }
                    case 'c': {
                        if (context >= 0) {
                            ++score;
                            continue block6;
                        }
                        return score;
                    }
                    default: {
                        throw new RuntimeException("invalid character '" + field + "' in field sequence: " + new String(this.tripleComparator.getFieldSeq()));
                    }
                }
            }
            return score;
        }

        public String toString() {
            return new String(this.getFieldSeq());
        }
    }

    private static class ImplicitStatementFilter
    implements RecordIterator {
        private final RecordIterator wrappedIter;

        public ImplicitStatementFilter(RecordIterator wrappedIter) {
            this.wrappedIter = wrappedIter;
        }

        @Override
        public byte[] next() throws IOException {
            byte flags;
            boolean explicit;
            byte[] result;
            while ((result = this.wrappedIter.next()) != null && (explicit = ((flags = result[16]) & 1) != 0)) {
            }
            return result;
        }

        @Override
        public void set(byte[] value) throws IOException {
            this.wrappedIter.set(value);
        }

        @Override
        public void close() throws IOException {
            this.wrappedIter.close();
        }
    }

    private static class ExplicitStatementFilter
    implements RecordIterator {
        private final RecordIterator wrappedIter;

        public ExplicitStatementFilter(RecordIterator wrappedIter) {
            this.wrappedIter = wrappedIter;
        }

        @Override
        public byte[] next() throws IOException {
            boolean toggled;
            byte flags;
            boolean explicit;
            byte[] result;
            while ((result = this.wrappedIter.next()) != null && (explicit = ((flags = result[16]) & 1) != 0) == (toggled = (flags & 8) != 0)) {
            }
            return result;
        }

        @Override
        public void set(byte[] value) throws IOException {
            this.wrappedIter.set(value);
        }

        @Override
        public void close() throws IOException {
            this.wrappedIter.close();
        }
    }
}

