/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.Collections.Generic;

namespace LocaleInpector
{

internal sealed class PoEntryCollection // PoFile
{

public enum Duplicates {
    Join,
    Split,
    Jump
}

// dont't use a Dictionary<> (we need to keep sort)
private List<string>  m_originalStrings = new List<string>();
private List<PoEntry> m_po              = new List<PoEntry>();
private Duplicates    m_join;

public PoEntryCollection(Duplicates method)
{
    m_join = method;
}

public bool AddEntry(string path, string key, string original, string translated, string comment)
{
    if (KeyBindingTest.IsKeyBinding(key)) {
        return false;
    }

    path = PoEntry.FormatRef(path).Replace(System.IO.Path.DirectorySeparatorChar, '/');
    path += "*" + key;

    if (original.Length == 0) {
        comment += @"--Original is a empty string!";
        //original = path; // msgid "locale/en-US/browser/chrome/ ...
    }

    original = Escape(original);
    translated = Escape(translated);


    if (m_originalStrings.Contains(original)) {
        switch (m_join) {
            case Duplicates.Join:
                PoEntry existingPo = m_po[m_originalStrings.IndexOf(original)];
                if (existingPo.Translated != translated) {
                    existingPo.AppendComment("Different translation used for " + path + ": " + translated);
                    existingPo.AppendInfo("fuzzy");
                }
                existingPo.AppendReference(path);
                existingPo.AppendComment(comment);
                return true;

            case Duplicates.Split:
                comment += System.Environment.NewLine + System.Environment.NewLine + "Original:" + System.Environment.NewLine + original;
                bool loop = true;
                int idx = 2;
                while (loop) {
                    string orig2 = original + "\\n" + "-- Translator: please remove this line [" + idx.ToString() + "] --";
                    if (m_originalStrings.Contains(orig2) == false) {
                        original = orig2;
                        loop = false;
                    }
                    idx++;
                }
                break;

            case Duplicates.Jump:
                return false;
        }
    }

    PoEntry newPo = new PoEntry();
    newPo.AppendReference(path);
    newPo.AppendOriginal(original);
    newPo.AppendTranslated(translated);
    newPo.AppendComment(comment);

    m_originalStrings.Add(original);
    m_po.Add(newPo);
    return true;
}

// A\common cause "for" the problem\r\n is using backslashes(\)
// A\\common cause \"for\" the problem\n is using backslashes(\\)
private static string Escape(string plainText)
{
    plainText = plainText.Replace("\\",    "\x00");
    plainText = plainText.Replace(@"\r\n", @"\n");
    plainText = plainText.Replace(@"\r",   @"\n");
    plainText = plainText.Replace("\"",    @"\""");
    plainText = plainText.Replace("\r\n",  @"\n");
    plainText = plainText.Replace("\r",    @"\n");
    plainText = plainText.Replace("\n",    @"\n");
    plainText = plainText.Replace("\x00",  @"\\");
    return plainText;
}


public void Serialize(System.IO.StringWriter writer)
{
    for (int idx = 0; idx < m_po.Count; idx++) {
        PoEntry entry = m_po[idx];

        // #: kwview.cc:2758 kwview.cc:2790 kwview.cc:2791
        // msgid "Change \"Custom\" Variable"
        // msgstr "Alterar variável personalizável"
        writer.WriteLine();
        List<string> comments = entry.Comment;
        if (comments != null) {
            foreach (string str in comments) {
                writer.WriteLine(@"#. {0}", str);
            }
        }
        writer.WriteLine(@"#: {0}", entry.References);
        // properties: c-format
        // #, fuzzy
        if (entry.Fuzzy) {
            writer.WriteLine(@"#, fuzzy");
        }
        //writer.WriteLine(@"#, no-c-format");
        writer.WriteLine(@"msgid ""{0}""", entry.Original);
        writer.WriteLine(@"msgstr ""{0}""", entry.Translated);
    }
}

}//class
}//ns
