/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.IO;
using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace LocaleInpector
{

internal class PoFile : NameValueFile
{

public PoFile(string path) : base(path)
{
}

public override void LoadContent()
{
    m_content = this.LoadText();
}

public override Merge GetMerge()
{
    return null;
}

public override Dictionary<string, string> LoadHashtableLogged()
{
    return this.LoadHashtable();
}

public override Dictionary<string, string> LoadHashtable()
{
    Dictionary<string, string> keys = new Dictionary<string, string>();
    LoadHashtableCore(keys);
    return keys;
}

public Dictionary<string, PoEntry> LoadHashtableII()
{
    List<PoEntry> arr = this.LoadHashtableCore2();
    Dictionary<string, PoEntry> table = new Dictionary<string, PoEntry>(arr.Count);

    for (int idx = 0; idx < arr.Count; idx++) {
        PoEntry poEntry = arr[idx];
        string id = poEntry.References;
        if (id == null) {
            continue;
        }
        if (id.Length == 0) {
            continue;
        }
        while (table.ContainsKey(id)) {
            id += " (repeated)";
        }
        table[id] = poEntry;
    }

    return table;
}

private void LoadHashtableCore(Dictionary<string, string> table)
{
    List<PoEntry> arr = this.LoadHashtableCore2();

    for (int idx = 0; idx < arr.Count; idx++) {
        PoEntry poEntry = arr[idx];
        string id = poEntry.References;
        if (id == null) {
            continue;
        }
        if (id.Length == 0) {
            continue;
        }
        while (table.ContainsKey(id)) {
            id += " (repeated)";
        }
        if (m_isTranslated) {
            if (poEntry.Translated != null) {
                table[id] = poEntry.Translated;
            }
            if (poEntry.TranslatedPlural != null) {
                table[id + " (plural)"] = poEntry.TranslatedPlural;
            }
        } else {
            if (poEntry.Translated != null) {
                table[id] = poEntry.Original;
            }
            if (poEntry.TranslatedPlural != null) {
                table[id + " (plural)"] = poEntry.OriginalPlural;
            }
        }
    }
}

private List<PoEntry> LoadHashtableCore2()
{
    string section = string.Empty;
    List<string> lines = Strings.SplitLines(m_content);

    string mode = null;
    List<PoEntry> arr = new List<PoEntry>();
    PoEntry entry = null;

    for (int idx = 0; idx < lines.Count; idx++) {

        string[] keyValue = ParseLine(lines[idx]);
        if (keyValue == null) {
            continue;
        }

        if (keyValue[0].Length == 0) {
            keyValue[0] = mode;
        }

        if (keyValue[0] != mode) {
            bool newEntry = false;

            if (entry == null) {
                //entry = new PoEntry();
                newEntry = true;
            } else {
                switch (keyValue[0]) {
                    case "#~":
                    case "#,":
                    case "#:":
                        newEntry = entry.Empty == false; //entry.References != null;
                        break;
                    case "msgid":
                        newEntry = entry.Original != null;
                        break;
                    case "msgid_plural":
                        newEntry = entry.OriginalPlural != null;
                        break;
                    case "msgstr":
                    case "msgstr[0]":
                        newEntry = entry.Translated != null;
                        break;
                    case "msgstr[1]":
                        newEntry = entry.TranslatedPlural != null;
                        break;
                    default:
                        newEntry = false;
                        break;
                }
            }

            if (newEntry) {
                entry = new PoEntry();
                arr.Add(entry);
            }
            mode = keyValue[0];
        }

        // #, fuzzy, c-format
        switch (mode) {
            case "#,":
                entry.AppendInfo(keyValue[1]);
                break;
            case "#:":
                entry.AppendReference(keyValue[1]);
                break;
            case "msgid":
                entry.AppendOriginal(keyValue[1]);
                break;
            case "msgid_plural":
                entry.AppendOriginalPlural(keyValue[1]);
                break;
            case "msgstr":
            case "msgstr[0]":
                entry.AppendTranslated(keyValue[1]);
                break;
            case "msgstr[1]":
                entry.AppendTranslatedPlural(keyValue[1]);
                break;
            default:
                break;
        }
    }

    UnEscape3(arr);
    return arr;
}

private static void UnEscape3(List<PoEntry> arr)
{
    for (int idx = 0; idx < arr.Count; idx++) {
        PoEntry entry = arr[idx];
        entry.UnEscape2();
    }
}

// #: import.label
// #: import.accesskey
// msgid "&Import..."
// msgstr ""

// #: kwview.cc:2758 kwview.cc:2790 kwview.cc:2791
// msgid "Change Custom Variable"
// msgstr "Mudar Varivel Personalizada"

private static string[] ParseLine(string line)
{
    string[] arr = ParseLineCore(line);
    if (arr == null) {
        return arr;
    }

    string txt = arr[1];

    arr[1] = txt;
    return arr;
}

public override string Sanitize(string txt)
{
    // del KDE access &key (preserving &entities;)
    if (txt.IndexOf('&') > -1) {
        if (Strings.CountSubstrings(txt, "&") == 1) {
            if (EntityTest.HasEntity(txt) == false) {
                txt = txt.Replace("&", string.Empty);
                return txt;
            }
        }
    }

    // del gnome access _key
    if (txt.IndexOf('_') > -1) {
        if (Strings.CountSubstrings(txt, "_") == 1) {
            txt = txt.Replace("_", string.Empty);
            return txt;
        }
    }

    return txt;
}

private static string[] ParseLineCore(string line)
{
    if (line == null) {
        return null;
    }

    line = line.Trim();

    if (line.Length < 2) {
        return null;
    }

    string[] ret = new string[2];

    if (line[0] == '#') {
        switch (line[1]) {
            case ':':
            case ',':
            case '~':
                ret[0] = line.Substring(0, 2);
                ret[1] = line.Substring(2).TrimStart();
                return ret;
            default:
                return null;
        }
        /*
        if (line[1] != ':') {
            return null;
        }
        ret[0] = "#:";
        ret[1] = line.Substring(2).TrimStart();
        return ret;
        */
    }

    // ""
    // msgstr ""
    // msgstr[0] "%s vence o jogo com %d ponto"
    // 01234567890
    if (line[line.Length - 1] != '"') {
        return null;
    }

    // "
    // msgstr "
    // msgstr[0] "%s vence o jogo com %d ponto
    string line2 = line.Substring(0, line.Length - 1);
    int idxQuote = line2.IndexOf('"');
    if (idxQuote == -1) {
        return null;
    }

    string[] val = line2.Split(new char[] { '"' }, 2);
    val[0] = val[0].TrimEnd();

    return val;
}

}//class
}//ns
